/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.widgets;

import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.events.*;

public class ScrollBar extends Widget {
	Scrollable parent;
	int minimum, maximum, selection, thumb, increment, pageIncrement;
	
ScrollBar (Scrollable parent, int style) {
	super(parent, style);
}

public void addSelectionListener (SelectionListener listener) {
	checkWidget();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	TypedListener typedListener = new TypedListener(listener);
	addListener (SWT.Selection,typedListener);
	addListener (SWT.DefaultSelection,typedListener);
}

void createWidget(Display display, Widget parent, int style, int index) {
	super.createWidget(display, parent, style, index);
	this.parent = (Scrollable) parent;
	_createHandle (parent, style, index);
}

public boolean getEnabled () {
	checkWidget ();
	return (state & DISABLED) == 0;
}

public int getIncrement () {
	checkWidget ();
	return increment;
}

public int getMaximum () {
	checkWidget ();
	return maximum;
}

public int getMinimum () {
	checkWidget ();
	return minimum;
}

public int getPageIncrement () {
	checkWidget ();
	return pageIncrement;
}

public Scrollable getParent () {
	checkWidget();
	return parent;
}

public int getSelection () {
	checkWidget ();
	return selection;
}

public Point getSize () {
	checkWidget ();
	return _getSize ();
}

public int getThumb () {
	checkWidget ();
	return thumb;
}

public boolean getVisible () {
	checkWidget ();
	return (state & HIDDEN) == 0;
}

public boolean isEnabled () {
	checkWidget();
	return getEnabled () && parent.isEnabled ();
}

public boolean isVisible () {
	checkWidget();
	return getVisible () && parent.isVisible ();
}

void releaseHandle () {
	super.releaseHandle ();
	parent = null;
}

void releaseParent () {
	super.releaseParent ();
	if (parent.horizontalBar == this) parent.horizontalBar = null;
	if (parent.verticalBar == this) parent.verticalBar = null;
}

public void removeSelectionListener (SelectionListener listener) {
	checkWidget();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (eventTable == null) return;
	eventTable.unhook (SWT.Selection, listener);
	eventTable.unhook (SWT.DefaultSelection,listener);	
}

public void setEnabled (boolean enabled) {
	checkWidget();
	if (((state & DISABLED) == 0) == enabled) return;
	state = enabled ? state & ~DISABLED : state | DISABLED;
	_setEnabled (enabled);
}

public void setIncrement (int value) {
	setValues(selection, minimum, maximum, thumb, increment, pageIncrement);
}

public void setMaximum (int value) {
	setValues(selection, minimum, maximum, thumb, increment, pageIncrement);
}

public void setMinimum (int value) {
	setValues(selection, minimum, maximum, thumb, increment, pageIncrement);
}

public void setPageIncrement (int value) {
	setValues(selection, minimum, maximum, thumb, increment, pageIncrement);
}

public void setSelection (int selection) {
	setValues(selection, minimum, maximum, thumb, increment, pageIncrement);
}

public void setThumb (int value) {
	setValues(selection, minimum, maximum, thumb, increment, pageIncrement);
}

public void setValues (int selection, int minimum, int maximum, int thumb, int increment, int pageIncrement) {
	checkWidget();
	if (minimum < 0) return;
	if (maximum < 0) return;
	if (thumb < 1) return;
	if (increment < 1) return;
	if (pageIncrement < 1) return;
	//TODO - more range checks
	this.selection = selection;
	this.minimum = minimum;
	this.maximum = maximum;
	this.thumb = thumb;
	this.increment = increment;
	this.pageIncrement = pageIncrement;
	_setValues (selection, minimum, maximum, thumb, increment, pageIncrement);
}

public void setVisible (boolean visible) {
	checkWidget ();
	if (((state & HIDDEN) == 0) == visible) return;
	if (visible) {
		sendEvent (SWT.Show);
		if (isDisposed ()) return;
	}
	state = visible ? state & ~HIDDEN : state | HIDDEN;
	_setVisible (visible);
	if (!visible) {
		sendEvent (SWT.Hide);
		if (isDisposed ()) return;
	}
}

/*---------------- COMMON INTERFACE ----------------*/

native void _createHandle (Widget parent, int style, int index) /*-{
	//TODO when a scroll bar is created, but the content doesn't go out of the bounds
	// then the scroll bar slider doesn't appear, to verify if it is possible/necessary 
	// to show it as is the usual SWT behaviour.  
	var jsParent = parent.@org.eclipse.swt.widgets.Widget::jsObject;
	var node = jsParent.scrollbarNode || jsParent.domNode;
	if((style & (@org.eclipse.swt.SWT::H_SCROLL)) != 0) {
		$wnd.swt.setCssStyle(node, {overflowX: "scroll"});
		if(node.style.overflowY != "scroll") {
			$wnd.swt.setCssStyle(node, {overflowY: "hidden"});
		}
	}
	if((style & (@org.eclipse.swt.SWT::V_SCROLL)) != 0) {
		$wnd.swt.setCssStyle(node, {overflowY: "scroll"});
		if(node.style.overflowX != "scroll") {
			$wnd.swt.setCssStyle(node, {overflowX: "hidden"});
		}
	} 
}-*/;

Point _getSize () {
	return (Point)callMethod("_getSize", "org.eclipse.swt.graphics.Point", null);
}

void _setEnabled (boolean enabled) {
	callMethod ("_setEnabled", null, new Object[]{new Boolean (enabled)});
}

void _setValues (int selection, int minimum, int maximum, int thumb, int increment, int pageIncrement) {
	Object[] args = new Object[]{
		new Integer(selection),
		new Integer(minimum),
		new Integer(maximum),
		new Integer(thumb),
		new Integer(increment),
		new Integer(pageIncrement),
	};
	callMethod("_setValues", null, args);
}

void _setVisible (boolean visible) {
	callMethod ("_setVisible", null, new Object[]{new Boolean (visible)});
}
}
